import unittest
import os

from scattool.packaging import PackageWorkspace

from test.support import TestCaseBase
from os import path

class TestPackaging(TestCaseBase):

    def test_create_workspace(self):
        workspace = PackageWorkspace(self.test_dir)
        self.assertIsNotNone(workspace)

    def test_create_workspace_raises_if_folder_missing(self):
        missing_folder = os.path.join(self.test_dir, "missing")
        self.assertRaises(Exception, lambda: PackageWorkspace(missing_folder))

    def test_add_remove_files(self):
        workspace = PackageWorkspace(self.test_dir)
        self.assertEqual(0, len(workspace.list_payload()))
        self.assertEqual(0, len(workspace.list_adjuncts()))

        sample_1 = self.sample_resource("sample_test1.txt")
        sample_2 = self.sample_resource("sample_test2.txt")        

        workspace.add_to_payload(sample_1)
        self.assertEqual(1, len(workspace.list_payload()))
        self.assertEqual(0, len(workspace.list_adjuncts()))

        self.assert_same_payload( [sample_1], workspace.list_payload())

        workspace.add_to_payload(sample_2)
        self.assertEqual(2, len(workspace.list_payload()))
        self.assertEqual(0, len(workspace.list_adjuncts()))

        self.assert_same_payload( [sample_1, sample_2], workspace.list_payload())

        workspace.remove_from_payload("sample_test1.txt")
        self.assertEqual(1, len(workspace.list_payload()))
        self.assertEqual(0, len(workspace.list_adjuncts()))

        self.assert_same_payload( [sample_2], workspace.list_payload())

        workspace.add_adjunct(sample_1)

        self.assert_same_payload( [sample_2], workspace.list_payload())
        self.assert_same_payload( [sample_1], workspace.list_adjuncts())
    
    def test_workspace_warnings(self):
        workspace = PackageWorkspace(self.test_dir)        
        self.assertFalse(workspace.processable())
        
        self.assert_same_entries( ["error.payload.empty"], workspace.errors())
        self.assert_same_entries( [], workspace.warnings())

        sample_1 = self.sample_resource("sample_test1.txt")  

        workspace.add_to_payload(sample_1)
        self.assertTrue(workspace.processable())
        
        self.assert_same_entries( [], workspace.errors())
        self.assert_same_entries( [], workspace.warnings())

        workspace.add_adjunct(sample_1)

        self.assert_same_entries( [], workspace.errors())
        self.assert_same_entries( ["warning.files_in_both_sets"], workspace.warnings())


    def test_workspace_temporary_persistence(self):
        workspace = PackageWorkspace(self.test_dir)
        self.assertFalse(workspace.processable())

        sample_1 = self.sample_resource("sample_test1.txt")
        sample_2 = self.sample_resource("sample_test2.txt")    
        
        workspace.add_to_payload(sample_1)
        workspace.add_adjunct(sample_2)  
        self.assertTrue(workspace.processable())

        # Safe to re-create from file system
        workspace = PackageWorkspace(self.test_dir)
        self.assert_same_payload( [sample_1], workspace.list_payload())
        self.assert_same_payload( [sample_2], workspace.list_adjuncts())
        self.assertTrue(workspace.processable())


