
from tkinter import *
from tkinter.constants import *
from tkinter import filedialog

from .widgets import ScrollableFrame, SimpleHeaderFrame, LocalizableElement

from pathlib import Path

import logging
logger = logging.getLogger(__name__)

class FilePicker(Frame):
    def __init__(self, container, value=None, actionlabel=None, summarylabel=None):
        super().__init__(container)
        self.summarylabel = summarylabel
        self.value = value

        e = Entry(self, textvariable=value, state="disabled")
        e.pack(side="left", fill=X, expand=True, padx=4, pady=4, ipadx=6, ipady=6)

        b = Button(self, textvariable=actionlabel, command=self.open_picker)
        b.pack(side="right")

    def open_picker(self):
        start_dir = Path.home
        if self.value.get():
            start_dir = Path(self.value.get()).parent

        file_name = filedialog.askopenfilename(
            title=self.summarylabel.get(),
            initialdir=start_dir
        )
        self.value.set(file_name)
    

class BindableFormScreen(SimpleHeaderFrame, LocalizableElement):

    def __init__(self, application, banner_label, **kwargs):
        LocalizableElement.__init__(self)
        self.banner_label = banner_label
        super().__init__(application, **kwargs)

    def heading_text(self):
        return self.textlabel(self.banner_label)

    def makeform(self, inner, header, fields):
        form = Frame(inner, highlightbackground="gray50", bg="gray50",
            highlightcolor="gray50", highlightthickness=5, pady=5, padx=10)
        form.pack(side="top", fill=X, expand=True, padx=0, pady=0)

        l = Label(form,
            textvariable=self.textlabel(header)
        )
        l.pack(side="top", fill=X, padx=0, pady=0)

        for (k, v) in fields.items():
            text = self.textlabel(k)
            f = Frame(form)
            f.pack(side="top", fill=X, expand=True)
            l = Label(f, padx=10, pady=10,  textvariable=text, width=20, anchor=E)
            l.pack(side="left", fill=X, padx=0, pady=0)
            if k.endswith("_file"):
                e = FilePicker(f, value=v, 
                                actionlabel=self.textlabel("file.browse"),
                                summarylabel=self.textlabel("%s_title" % k))
            else:
                e = Entry(f, textvariable=v)
            e.pack(side="left", fill=X, expand=True, padx=4, pady=4, ipadx=6, ipady=6)

        btns = Frame(form, pady=20, padx=10)
        save_btn = Button(btns, text="Save", command=self.on_save)
        save_btn.pack(side=RIGHT, padx=5)
        btns.pack(side="top", fill=BOTH, expand=True)


class ConfigurationStartScreen(BindableFormScreen):

    def __init__(self, application):
        super().__init__(application, "config.inst_header", enable_back=False)

    def build_body(self, inner):
        self.name_field = StringVar()
        self.namespace_field = StringVar()
        self.archon_field = StringVar()

        self.loadconfig()

        fields = {
            "config.inst_name" : self.name_field,
            "config.inst_namespace" : self.namespace_field,
            "config.inst_code" : self.archon_field
        }

        self.makeform(inner, "config.inst_summary", fields)

    def loadconfig(self):
        orgconfig = self.application.configuration.orgconfig
        self.name_field.set(orgconfig.name)
        self.namespace_field.set(orgconfig.namespace)
        self.archon_field.set(orgconfig.archon_code)

    def on_save(self):
        orgconfig = self.application.configuration.orgconfig
        orgconfig.name = self.name_field.get()
        orgconfig.namespace = self.namespace_field.get()
        orgconfig.archon_code = self.archon_field.get()
        self.application.configuration.set_orgconfig(orgconfig)
        if self.application.profiles.firstrun:
            logger.info("Users not yet intitiatised, launching welcome screen")
            self.application.show_screen("new_profile", forceadmin=True, createanother=True, disablecancel=True)
        else:
            logger.info("Presenting user picker")
            self.application.show_screen("select_profile")


class EncryptionConfigScreen(BindableFormScreen):

    def __init__(self, application):
        super().__init__(application, "config.encryption_header", enable_back=False)

    def build_body(self, inner):
        self.passphrase_field = StringVar()
        
        self.loadconfig()

        fields = {
            "config.passphrase_file" : self.passphrase_field
        }

        self.makeform(inner, "config.inst_summary", fields)

    def loadconfig(self):
        config = self.application.configuration.encryption
        self.passphrase_field.set(config.passphrase_file)

    def on_save(self):
        config = self.application.configuration.encryption
        config.passphrase_file = self.passphrase_field.get()
        logger.info("Setting passphrase_file to %s", config.passphrase_file)
        self.application.configuration.set_encryption(config)