import os
import json
import shutil

import subprocess

from os import error, path
from shutil import Error, rmtree

import pathlib
from pathlib import Path
from tkinter import Pack

from .util import UidMinter

from ..scatlib import Scatlib

from .. import sbagit

import logging
logger = logging.getLogger(__name__)

import tarfile, zipfile

import re
from .util import ChangeNotifier

import bagit

from tempfile import TemporaryDirectory


## standard sbag-info.txt metadata
STANDARD_PACKAGE_INFO_HEADERS = [
    "Source-Organization",
    "Organization-Address",
    "Contact-Name",
    "Contact-Phone",
    "Contact-Email",
    "External-Description",
    "External-Identifier",
    "Bag-Size",
    "Bag-Group-Identifier",
    "Bag-Count",
    "Internal-Sender-Identifier",
    "Internal-Sender-Description",
    "BagIt-Profile-Identifier",
    # Bagging-Date is autogenerated
    # Payload-Oxum is autogenerated
]

class PackageWorkspace:

    def __init__(self, application_workspace, configuration):
        self.application_workspace = application_workspace
        self.base_folder = application_workspace.currentpackage
        self.base_path = Path(self.base_folder)
        if not(path.isdir(self.base_folder)):
            raise Error("Folder %s did not exist" % self.base_folder)
        self.assembly = PackageAssembly()
        self.configuration = configuration
        self.change_listeners = ChangeNotifier()
        self.isopen = False
        self.underlying_package = None

    def can_resume(self):
        folder_listing = os.listdir(self.base_folder)
        return ("state.json" in folder_listing)

    def resume(self):
        self.rehydrate_state()

    def add_change_listener(self, onchange):
        self.change_listeners.add_listener(onchange)

    def list_payload(self):
        return self.assembly.payload

    def list_adjuncts(self):
        return self.assembly.adjuncts

    def add_to_payload(self, path):
        p = Path(path)
        entry = {
            "src": str(Path(path).absolute()),
            "path": str(p.name),
            "folder": "",
            "type": "FILE",
            "name": p.name
        }
        self.assembly.add_payload(entry)
        self.persist_state()

    def add_folder_to_payload(self, path):
        folder = Path(path)
        entry = {
            "src": str(Path(path).absolute()),
            "path": str(folder.name),
            "folder": "",
            "type": "FOLDER",
            "name": folder.name
        }
        self.assembly.add_payload(entry)
        for item in Path(path).rglob('*'):
            if (os.path.isfile(item)):
                rel_path = item.relative_to(folder)
                entry = {
                    "src": str(item.absolute()),
                    "path": str(rel_path),
                    "folder": str(folder.absolute()),
                    "type": "FILE",
                    "name": item.name
                }
                self.assembly.add_payload(entry)
        self.persist_state()

    def check_package(self):
        assembly = self.assembly
        if assembly:
            if (len(assembly.payload) > 0 and len(assembly.metadata) >= 5):
                all_non_empty = True
                for key, val in assembly.metadata.items():
                    if not val:
                        all_non_empty = False
                return all_non_empty
        return False

    def set_meta(self, key, value):
        assembly = self.assembly
        if value:
            assembly.metadata[key] = value

    def get_meta(self, key):
        assembly = self.assembly
        return assembly.metadata.get(key, "")

    def remove_from_payload(self, path):
        assembly = self.assembly
        folder_full_path = ""
        if path in assembly.payload:
            folder_full_path = assembly.payload[path]['src']
            assembly.payload.pop(path)
        if path in assembly.adjuncts:
            assembly.adjuncts.pop(path)
        for k in list(assembly.payload.keys()):
            if assembly.payload[k]['folder'] == folder_full_path:
                assembly.payload.pop(k)
        self.persist_state()

    def remove_all_payload(self):
        assembly = self.assembly
        assembly.payload = {}
        assembly.adjuncts = {}
        self.persist_state()

    def start_new_package(self):
        self.assembly = PackageAssembly()
        self.persist_state()

    def add_adjunct(self, path):
        p = Path(path)
        entry = {
            "src": str(Path(path).absolute()),
            "path": str(p.name),
            "folder": "",
            "name": p.name
        }
        self.assembly.add_adjunct(entry)
        self.persist_state()

    def remove_adjunct(self, path):
        self.assembly.adjuncts.pop(path)
        self.persist_state()

    def processable(self):
        return not(self.errors())

    def warnings(self):
        _warnings = []
        assembly = self.assembly
        if (assembly.payload.keys() & assembly.adjuncts.keys()):
            _warnings.append("warning.files_in_both_sets")
        return _warnings

    def errors(self):
        _errors = []
        assembly = self.assembly
        if not(assembly.payload):
            _errors.append("error.payload.empty")
        return _errors

    def describe_state(self):
        assembly = self.assembly
        if assembly.notempty():
            return {
                "payload": assembly.payload,
                "adjuncts": assembly.adjuncts,
                "metadata": assembly.metadata
            }
        else:
            return None

    def persist_state(self):
        workspace_state = path.join(self.base_folder, "state.json")
        with open(workspace_state, 'w', encoding='utf-8') as f:
            state = self.describe_state()
            if state:
                f.write(json.dumps(state))
        self.change_listeners.notify()

    def clear_state(self):
        workspace_state = self.base_path / "state.json"
        if workspace_state.exists():
            workspace_state.unlink()

    def rehydrate_state(self):
        workspace_state = path.join(self.base_folder, "state.json")
        with open(workspace_state, 'r', encoding='utf-8') as f:
            c = f.read()
            state = json.loads(c)
            if state:
                self.assembly = PackageAssembly()
                for item in state["payload"]:
                    self.assembly.payload[item] = state["payload"].get(item)
                for item in state["adjuncts"]:
                    self.assembly.adjuncts[item] = state["adjuncts"].get(item)
                self.assembly.metadata = state.get("metadata", {})

    def uid_minter(self):
        orgconfig = self.configuration.orgconfig
        logger.debug("Seeding UID Minter with %s", orgconfig.namespace)
        return UidMinter(orgconfig.namespace, version=5)

    def pass_phrase_file_path(self):
        pass_phrase_path = Path(self.configuration.encyption.pass_phrase_file)
        logger.debug("Using pass_phrase_path %s", pass_phrase_path)
        return pass_phrase_path

    def noop_log(str):
        pass

    def switch_profile(self, current_profile):
        self.current_profile = current_profile
        logger.info("Setup profile as %s", self.current_profile)

    def save_as_aip(self, encryption=False, double_bag=False, reporter=noop_log):
        logger.info("Saving as AIP encryption=%s, double_bag=%s", encryption, double_bag)
        self.name = self.uid_minter().uuid()
        base = Path(self.base_folder)
        tmpdir = base / str(self.name)
        tmpdir.mkdir(parents=True, exist_ok=True)
        logger.info("Using tmpdir %s", tmpdir)
        reporter("Using packaging folder %s" % tmpdir)
        

        ##  populate payload bagging directory
        for (localpath, item) in self.assembly.payload.items():
            path = Path(localpath)
            dst = tmpdir / path
            dst.parent.mkdir(parents=True, exist_ok=True)
            src_path = Path(item["src"])
            if src_path.is_file():
                reporter("Copying %s to packaging folder" % src_path)
                ##  copy payload file to payload bagging directory
                shutil.copyfile(src_path, dst)
            elif src_path.is_dir():
                logger.info("Ignoring directory %s", src_path)
                ##  copy payload directory to payload bagging directory
                #shutil.copytree(src_path, dst)
            else:
                ## what happens to sym links?
                logger.error("is error")
                pass
        adjuncts = []
        for (localpath, item) in self.assembly.adjuncts.items():
            path = Path(localpath)
            dst = tmpdir / path
            dst.parent.mkdir(parents=True, exist_ok=True)
            src_path = Path(item["src"])
            adjuncts.append(src_path)
            reporter("Using %s as an adjunct file" % src_path)

        current_profile = self.current_profile
        organisationinfo = self.configuration.orgconfig

        package_metadata = self.assembly.assemble_package_metadata(current_profile, organisationinfo)
        logger.info("Assembled package metadata: %s", package_metadata)

        options = {
            "double_bag": double_bag,
            "payload": tmpdir,
            "adjunct_files": adjuncts,
            "metadata": package_metadata,
            "tar": True,
            "show_suffix": False,
            "encryption": encryption
        }

        if encryption:
            reporter("Enabling encryption")
            encryption_settings = self.configuration.encryption
            options["encryption_algorithm"] = "AES256"
            options["pass_phrase_file_name"] = Path(encryption_settings.passphrase_file)

        msg = "Making package. Using the following settings:\n"
        for x in options.items():
            msg += ("  %s: %s\n" % x )
        reporter(msg)


        package = Scatlib.MakePackage(**options)

        logger.info("Completed packaging. Result was %s", package)
        packagepath = package.name
        self.application_workspace.localstore.add_aip(packagepath)
        logger.info("Removing tmpdir %s", tmpdir)
        tmpdir.rmdir()

        reporter("Created AIP %s" % packagepath.name)


    def save_as_sip(self):
        pass

    def save_as_dip(self, reporter=noop_log):
        logger.info("Saving as DIP")
        self.name = self.uid_minter().uuid()
        base = Path(self.base_folder)
        tmpdir = base / str(self.name)
        tmpdir.mkdir(parents=True)
        logger.info("Using tmpdir %s", tmpdir)

        datadir = tmpdir / "data"

        assembly = self.assembly

        for (localpath, item) in assembly.payload.items():
            path = Path(localpath)
            dst = datadir / path
            dst.parent.mkdir(parents=True, exist_ok=True)
            src_path = Path(item["src"])
            if src_path.is_file():
                ##  copy payload file to payload bagging directory
                shutil.copyfile(src_path, dst)
            elif src_path.is_dir():
                logger.info("Ignoring directory %s", src_path)
            else:
                ## what happens to sym links?
                logger.error("is error")
                pass

        for (localpath, item) in self.assembly.adjuncts.items():
            path = Path(localpath)
            dst = tmpdir / path
            src_path = Path(item["src"])
            shutil.copyfile(src_path, dst)

        with open(tmpdir / "metadata.json", 'w', encoding='utf-8') as f:
            f.write(json.dumps(assembly.metadata))

        base_name = tmpdir
        root_dir = base_name.parent.resolve()
        base_dir = base_name.name

        suffix = "zip"

        file_name = shutil.make_archive(
            base_name=base_name,
            format=suffix,
            root_dir=root_dir,
            base_dir=base_dir,
        )

        logger.info("Completed packaging. Result was %s", file_name)
        self.application_workspace.localstore.add_dip(Path(file_name))
        logger.info("Removing tmpdir %s", tmpdir)
        shutil.rmtree(tmpdir)

        reporter("Created DIP %s" % file_name)

    def get_fixity(self):
        return self.fixity

    def load_package_into_workspace(self):
        logger.info("Loading current package into workspace")
        if not self.underlying_package:
            raise Error("Can't load a package when no package is open")
        src_package = self.underlying_package
        logger.info("Using base location %s", self.base_folder)
        logger.info("Source package id %s", src_package.name)
        scratch_path = self.application_workspace.tmppath / src_package.name
        if scratch_path.exists():
            shutil.rmtree(scratch_path)
        scratch_path.mkdir()
        logger.info("Unpack source package to %s", scratch_path)
        exploded = src_package.unpack_to(scratch_path)

        self.start_new_package()
        for adjunct_file in src_package.adjunct_files:
            logger.info("Adding adjunct into workspace: %s", adjunct_file)
            self.add_adjunct(exploded / adjunct_file)

        data_folder = exploded / "data"
        logger.info("Add folder %s", data_folder)
        for item in data_folder.glob("*"):
            if (item.is_dir()):
                self.add_folder_to_payload(item)
            else:
                self.add_to_payload(item)
        self.assembly.metadata = self.underlying_package.metadata

    def close_package(self):
        src_package = self.underlying_package
        if src_package:
            scratch_path = self.application_workspace.tmppath / src_package.name
            if scratch_path.exists():
                shutil.rmtree(scratch_path)
        self.underlying_package = None
        self.clear_state()

    def open_package(self, file_path):
        packagepath = Path(file_path)
        if not packagepath.is_file():
            return None
        _package = Package(self, packagepath)
        _package.unpack()
        self.underlying_package = _package
        return _package

class PackageAssembly():

    def __init__(self) -> None:
        self.payload = {}
        self.adjuncts = {}
        self.metadata = {}

    def list_payload(self):
        return self.payload

    def list_adjuncts(self):
        return self.adjuncts

    def add_payload(self, entry):
        key = entry["path"]
        if (entry["type"] == "FOLDER"):
            key = key + "/"
        self.payload[key] = entry

    def add_adjunct(self, entry):
        self.adjuncts[entry["path"]] = entry

    def notempty(self):
        return self.adjuncts or self.payload or self.metadata

    def assemble_package_metadata(self, current_profile, orginfo):
        package_metadata = {}

        ##  initialise package metadata
        for tag in STANDARD_PACKAGE_INFO_HEADERS:
            package_metadata[tag] = ""        
        logger.info("Initialising package metadata as %s", package_metadata)
        
        ## Seed user and org information
        institution_name = orginfo.name
        package_metadata["Source-Organization"] = orginfo.name
        package_metadata["Contact-Name"] = current_profile.name

        archon_code = orginfo.archon_code

        descriptive_metadata = {}
        for (tag, val) in self.metadata.items():
            ##  only tags 'title', 'catalogue reference', 'accession reference'
            ##  'access restriction' and 'use restriction' are recognized
            if tag == "title":
                package_metadata["External-Description"] = val
                descriptive_metadata[tag] = val
            elif tag == "catalogue_reference":
                external_identifier = "%s:%s" % (archon_code, val)
                package_metadata["External-Identifier"] = external_identifier
                descriptive_metadata[tag] = val
            elif tag == "accession_reference":
                package_metadata["Internal-Sender-Identifier"] = val
                descriptive_metadata[tag] = val
            elif tag == "access_restriction":
                descriptive_metadata[tag] = val
            elif tag == "use_restriction":
                descriptive_metadata[tag] = val
            else:
                logger.warn("Unexpected metadata tag %s", tag)

        package_metadata["Internal-Sender-Description"] = json.dumps(descriptive_metadata)
        return package_metadata

class Package():

    def __init__(self, packaging_workspace, packagepath):
        self.packaging_workspace = packaging_workspace
        self.packagepath = packagepath
        self.name = packagepath.name
        self.encrypted = False
        self.inner_payload = "?"
        self.passphrase_file = packaging_workspace.configuration.encryption.passphrase_file

    def unpack(self):
        sbag = sbagit.SBag(self.packagepath)
        self.sbag = sbag
        self.valid = sbag.is_valid()

        unpack = {}
        if sbag.is_valid():
            p = Path(sbag.path)
            unpack["name"] = p.name
            unpack["tags"] = sbag.tags
            unpack["algorithms"] = sbag.algorithms
            unpack["container"] = sbag.container
            unpack["metadata"] = sbag.info
            unpack["entries"] = sbag.entries
            unpack["adjunct"] = sbag.adjunct
            unpack["payload"] = sbag.payload

        fixity = {}
        for key in unpack["entries"].keys():
            if re.match("manifest-", key) or re.match("tagmanifest-", key):
                pass
            elif key == "bag-info.txt" or key == "bagit.txt":
                pass
            else:
                values = unpack["entries"][key]
                fixity[key.replace("data/", "", 1)] = values

        unpack["fixity"] = fixity
        self.unpacked = unpack
        self.outercontainer = sbag.container
        self.metadata = sbag.info
        self.bagit_tags = sbag.tags
        self.algorithms = sbag.algorithms
        self.fixity = fixity

        payload_files = sbag.payload
        payload_count = len(payload_files)

        if payload_count == 0:
            self.warnings.append("empty.payload")
        elif payload_count == 1 and next(iter(payload_files.keys())) == self.name:
            self.doublebagged = True
        else:
            self.doublebagged = False

        if self.doublebagged:
            self.inner_serialisation = "?"

        self.payload_count = payload_count
        self.payload_files = payload_files
        self.adjunct_files = sbag.adjunct
        self.adjunct_count = len(self.adjunct_files)

    def read_inner_bag(self):
        with TemporaryDirectory() as temp_dir:
            tmp_path = Path(temp_dir)
            inner_bag_file = self.__extract_inner_bag_to(tmp_path)
            self.__identify_inner_bag(inner_bag_file)

    def __extract_inner_bag_to(self, target_path):
        inner_bag_entry = "%s/data/%s" % (self.name, self.name)
        logger.info("Reading inner bag. Entry %s", inner_bag_entry)
        if self.outercontainer == "tar":
            tar = tarfile.open(self.packagepath)
            tar.extract(inner_bag_entry, path=target_path)
        elif self.outercontainer == "zip":
            zip = zipfile.open(self.packagepath)
            zip.extract(inner_bag_entry, path=target_path)
        else:
            raise Exception("Cannot unpack item in %s" % self.outercontainer)
        return (target_path / inner_bag_entry)

    def __decrypt_inner_bag(self, inner_bag_file):
        plain_file = inner_bag_file.with_suffix(inner_bag_file.suffix + ".plain")
        gpg = subprocess.run(
            [
                "gpg",
                "--batch",
                "--output",
                plain_file,
                "--passphrase-file",
                self.passphrase_file,
                "-d",
                inner_bag_file,
            ]
        )
        if gpg.returncode:
            print(gpg)
            print("###### fail #######")
            raise Error("Decrypt fail")
        return plain_file

    def __identify_inner_bag(self, inner_bag_file, decrypted=False):
        self.encrypted = decrypted
        if tarfile.is_tarfile(inner_bag_file):
            self.__list_inner_tar(inner_bag_file)
        elif zipfile.is_zipfile(self.name):
            self.__list_inner_zip(inner_bag_file)
        elif not decrypted:
            plain_file = self.__decrypt_inner_bag(inner_bag_file)
            self.__identify_inner_bag(plain_file, decrypted=True)
        else:
            raise Error("Could not read inner bag")

    def __list_inner_tar(self, inner_bag_file):
        self.inner_serialisation = "tar"
        self.inner_listing = [m.name for m in tarfile.open(inner_bag_file).getmembers()]
        base_path = "%s/data/" % self.name
        self.inner_payload = [f.replace(base_path, "", 1) for f in self.inner_listing if f.startswith(base_path)]

    def __list_inner_zip(self, inner_bag_file):
        self.inner_serialisation = "zip"
        self.inner_listing = [m.name for m in zipfile.open(inner_bag_file).getmembers()]
        base_path = "%s/data/" % self.name
        self.inner_payload = [f.replace(base_path, "", 1) for f in self.inner_listing if f.startswith(base_path)]

    def unpack_to(self, target_path):
        logger.info("Unpacking to %s", target_path)
        print(self)
        data_container = None
        if self.outercontainer == "tar" and not self.doublebagged:
            data_container = self.packagepath
        elif self.outercontainer == "tar" and self.doublebagged:
            inner = target_path / "inner"
            logger.info("Extracting inner payload")
            inner_bag_file = self.__extract_inner_bag_to(inner)
            logger.info("Unpacked %s", inner_bag_file)
            if tarfile.is_tarfile(inner_bag_file):
                data_container = inner_bag_file
            elif self.encrypted:
                plain_file = self.__decrypt_inner_bag(inner_bag_file)
                logger.info("Decrypted as %s", plain_file)
                data_container = plain_file
            else:
                data_container = inner_bag_file
        else:
            raise Error("Cannot unpack package")
        inflated = target_path / "inflated"
        container = tarfile.open(data_container)
        container.extractall(path=inflated)
        logger.info("Inflated to: %s", inflated)
        return inflated / self.name


    def report(self):
        lines = []
        lines.append("Name: %s" % self.name)
        lines.append("Valid: %s" % self.valid)
        lines.append("Bagit Tags: %s" % self.bagit_tags)
        lines.append("Container: %s" % self.outercontainer)
        lines.append("Algorithms: %s" % self.algorithms)
        lines.append("DoubleBagged: %s" % self.doublebagged)
        lines.append("Payload: (%d files)" % self.payload_count)
        for item in self.payload_files:
            lines.append("  - %s" % item )
        if self.doublebagged:
            lines.append("Inner bag encrypted: %s" % self.encrypted )
            lines.append("Inner bag container: %s" % self.inner_serialisation )
            lines.append("Inner bag payload (%d files):" % len(self.inner_payload) )
            for item in self.inner_payload:
                lines.append("  - %s" % item )
        lines.append("Adjuncts: (%d files)" % self.adjunct_count)
        for item in self.adjunct_files:
            lines.append("  - %s" % item )
        lines.append("Archival Metadata:")
        for k, v in self.metadata.items():
            lines.append("  %s: %s" % (k,v) )
        lines.append("Fixity statement:")
        for name, item_fixity in self.fixity.items():
            if self.doublebagged:
                lines.append(" %s (inner bag):" % name)
            else:
                lines.append(" %s:" % name)
            for k, v in item_fixity.items():
                lines.append(" - %s: %s" % (k,v) )

        return "\n".join(lines)
