import shutil, tempfile, os
import unittest
import bagit
import hashlib

from test.support import TestCaseBase
from os import path

class TestExternals(TestCaseBase):

    # For these tests, we'll create a bag with one file <demo.txt> with defined content
    DEMO_CONTENT = "ABC_123"

    def test_bagit_in_place(self):
        # Correctness of the bagit package is assumed. This test is here to sanity check that the dependency is
        # present and api does what we expect

        self.create_sample_bag()

        # Check bag structure correct
        listing = self.list_test_dir()
        self.assertEqual(7, len(listing))
        self.assertIn("data", listing)
        self.assertIn("bag-info.txt", listing)
        self.assertIn("bagit.txt", listing)
        self.assertIn("data", listing)
        self.assertIn("manifest-sha256.txt", listing)
        self.assertIn("manifest-sha512.txt", listing)
        self.assertIn("tagmanifest-sha512.txt", listing)

        # Check that original file moved into /data path
        inner_listing = os.listdir(path.join(self.test_dir, 'data'))
        self.assertEqual(1, len(inner_listing))
        self.assertIn("demo.txt", inner_listing)

        # Check file contents is correct
        self.assert_file_contents(self.DEMO_CONTENT, path.join(self.test_dir, 'data', 'demo.txt'))

        # Check hash is correct
        expected_sha256 = self.expected_sha256()
        self.assert_manifest_contents(expected_sha256, "data/demo.txt", path.join(self.test_dir, 'manifest-sha256.txt'))

    def test_bagit_validate(self):
        # If a payload file is modified, then the bag ceases to be valid
        bag = self.create_sample_bag()
        self.assertTrue(bag.is_valid())
        payload_file = path.join(self.test_dir, 'data', 'demo.txt')
        with open(payload_file, 'a') as f:
            f.write("xx")
        self.assert_file_contents("ABC_123xx", path.join(self.test_dir, 'data', 'demo.txt'))
        self.assertFalse(bag.is_valid())

    
    def test_bagit_api(self):
        # If a payload file is modified, then the bag ceases to be valid
        bag = self.create_sample_bag()
        self.assertSameEntries(["sha256", "sha512"], bag.algorithms)
        self.assertEqual("UTF-8", bag.encoding)
        expected_sha256 = self.expected_sha256()
        self.assertEqual(expected_sha256, bag.payload_entries()["data/demo.txt"]['sha256'])

    # Utilities

    def create_sample_bag(self):
        self.write_sample_file("demo.txt", self.DEMO_CONTENT)
        # Check folder structure has one file
        listing = self.list_test_dir()
        self.assertEqual(1, len(listing))
        self.assertEqual("demo.txt", listing[0])
        # make bag
        return bagit.make_bag(self.test_dir, {'MyMetadata': 'My value'})

    def expected_sha256(self):
        m = hashlib.sha256()
        m.update(self.DEMO_CONTENT.encode('UTF-8'))
        return m.hexdigest()
