from tkinter import *
from tkinter.constants import *
from tkinter import messagebox


from .widgets import ScrollableFrame, SimpleHeaderFrame

import logging
logger = logging.getLogger(__name__)

class ProfileFrame(Frame):
    def __init__(self, parent, profile, select_fn=None, edit_fn=None, delete_fn=None):
        self.bg_color = "steel blue"
        self.profile = profile

        cursor = "hand2" if select_fn else None

        Frame.__init__(self, parent, bg=self.bg_color, height=100, width=400, cursor=cursor)
        
        current_user = Label(self, text=profile.name, fg="white", bg=self.bg_color, padx=5, anchor=W)
        current_user.place(y=6)

        profile = Label(self, text=profile.org_label, fg="white", bg=self.bg_color, padx=5, anchor=W)
        profile.config(font=("Helvetica", 10))
        profile.place(y=36)

        if edit_fn:
            edit_btn = Button(self, relief=FLAT, highlightthickness=0, text="Edit", command=lambda: edit_fn(self.profile))
            edit_btn.place(y=10, x=310)

        if delete_fn:
            del_btn = Button(self, relief=FLAT, highlightthickness=0, text="Delete", command=lambda: delete_fn(self.profile))
            del_btn.place(y=50, x=310)

        if select_fn:
            on_click = lambda x: select_fn(self.profile)
            self.bind("<Button-1>", on_click)
            profile.bind("<Button-1>", on_click)
            current_user.bind("<Button-1>", on_click)
        
        self.current_user = current_user
        self.org_label = profile


    def refresh_data(self, profile):
        self.profile = profile
        self.current_user.configure(text=profile.name)
        self.org_label.configure(text=profile.org_label)        


class ProfilePicker(SimpleHeaderFrame):

    def __init__(self, application, enable_back=True, label="Change profile", enabled_edit=False, enable_select=False, users_only=False):
        self.label = label
        self.enabled_edit = enabled_edit
        self.enable_select = enable_select
        self.users_only = users_only
        self.widgets = {}
        SimpleHeaderFrame.__init__(self, application, enable_back=enable_back)

    def build_body(self, body):
        scrollingFrame = ScrollableFrame(body, lambda: 400)
        self.scrollingFrame = scrollingFrame
        scrollingFrame.pack(expand=True, fill=BOTH)
        self.inner = scrollingFrame.scrollable_frame
        self.profile_list = Frame(self.inner)
        profiles = self.list_profiles()
        for profile in profiles:
            if self.enabled_edit:
                widget = ProfileFrame(self.inner, profile, edit_fn=self.edit_profile, delete_fn=self.delete_profile)
            else:
                widget = ProfileFrame(self.inner, profile, select_fn=self.on_select_profile)
            widget.pack(pady=10, padx=10, expand=True, fill=Y, anchor=CENTER)
            self.widgets[profile.id] = widget

    def list_profiles(self):
        if self.users_only:
            return self.application.profiles.list_users()
        else:
            return self.application.profiles.list_all()

    def on_back_button(self):
        self.application.show_screen("main")

    def heading_text(self):
        return self.label

    def on_select_profile(self, profile):
        logger.info("Selecting profile %s", profile)
        self.application.switch_profile(profile)
        self.application.show_screen("main")

    def edit_profile(self, profile):
        logger.info("edit profile")
        self.application.show_screen("edit_profile", profile=profile)

    def delete_profile(self, profile):
        logger.info("remove profile")
        self.application.profiles.remove(profile)
        self.application.edit_profiles()
        self.widgets[profile.id].pack_forget()

    def present(self, admin=False):
        profiles = self.list_profiles()
        logger.info("Listed profiles: %s", profiles)
        for profile in profiles:
            if (profile.id in self.widgets):
                self.widgets[profile.id].refresh_data(profile)
            else:
                widget = ProfileFrame(self.inner, profile, self.on_select_profile, self.edit_profile, self.delete_profile)
                widget.pack(pady=10, padx=10, expand=True, fill=Y, anchor=CENTER)
                self.widgets[profile.id] = widget
        super().present()

class CreateProfileScreen(SimpleHeaderFrame):

    def __init__(self, application):
        SimpleHeaderFrame.__init__(self, application)

    def on_back_button(self):
        self.application.show_screen("main")

    def heading_text(self):
        return "Create profile"

    def build_body(self, inner):
        form = Frame(inner, highlightbackground="gray50", bg="gray50",
            highlightcolor="gray50", highlightthickness=5, pady=5, padx=10)
        form.pack(side="top", fill=X, expand=True, padx=0, pady=0)

        self.name_field = StringVar()
        self.role_field = StringVar()
        self.admin_field = IntVar(value=1)


        fields = {
            "Name" : self.name_field,
            "Role" : self.role_field
        }

        for (k, v) in fields.items():
            f = Frame(form)
            f.pack(side="top", fill=X, expand=True)
            l = Label(f, padx=10, pady=10,  text=k, width=20, anchor=E)
            l.pack(side="left", fill=X, padx=0, pady=0)
            e = Entry(f, textvariable=v)
            e.pack(side="left", fill=X, expand=True, padx=4, pady=4, ipadx=6, ipady=6)

        f = Frame(form)
        f.pack(side="top", fill=X, expand=True)
        l = Label(f, padx=10, pady=10,  text="Admin", width=20, anchor=E)
        l.pack(side="left", fill=X, padx=0, pady=0)
        e = Checkbutton(f, variable=self.admin_field)
        e.pack(side="left", anchor=W)

        btns = Frame(form, pady=20, padx=10)
        save_btn = Button(btns, text="Save", command=self.on_save)
        cancel_btn = Button(btns, text="Cancel", command=self.on_cancel)
        another_btn = Button(btns, text="Save and create another", command=self.on_another)
        another_btn.pack(side=RIGHT, padx=5)
        save_btn.pack(side=RIGHT, padx=5)
        cancel_btn.pack(side=RIGHT, padx=5)
        btns.pack(side="top", fill=BOTH, expand=True)


        self.adminCheck = e
        self.cancel_btn = cancel_btn

    def on_save(self):
        profile = self.application.profiles.new_profile()
        profile.name = self.name_field.get()
        profile.role = self.role_field.get()
        profile.admin = self.admin_field.get() or self.forceadmin
        self.application.profiles.save(profile)
        self.application.edit_profiles()

    def on_another(self):
        profile = self.application.profiles.new_profile()
        profile.name = self.name_field.get()
        profile.role = self.role_field.get()
        profile.admin = self.admin_field.get()
        self.application.profiles.save(profile)
        self.application.show_screen("new_profile", createanother=True)

    def on_cancel(self):
        self.application.edit_profiles()

    def present(self, forceadmin=False, createanother=False, disablecancel=False, disableback=False):
        super().present()
        self.name_field.set("")
        self.role_field.set("")
        self.admin_field.set(0)
        self.adminCheck["state"] = "normal"
        self.cancel_btn["state"] = "normal"
        self.forceadmin = forceadmin
        if forceadmin:
            self.admin_field.set(1)
            self.adminCheck["state"] = "disabled"
        if disablecancel:
            self.cancel_btn["state"] = "disabled"
        if disableback and self.backbutton:
            self.backbutton["state"] = "disabled"
    
class EditProfileScreen(SimpleHeaderFrame):

    def __init__(self, application):
        SimpleHeaderFrame.__init__(self, application)

    def heading_text(self):
        return "Edit profile"

    def build_body(self, inner):
        # scrollingFrame = ScrollableFrame(body, lambda: 600)
        # self.scrollingFrame = scrollingFrame
        # scrollingFrame.pack(expand=True, fill=BOTH)
        # inner = scrollingFrame.scrollable_frame

        form = Frame(inner, highlightbackground="gray50", bg="gray50",
            highlightcolor="gray50", highlightthickness=5, pady=5, padx=10)
        form.pack(side="top", fill=X, expand=True, padx=0, pady=0)

        self.name_field = StringVar()
        self.role_field = StringVar()
        self.admin_field = IntVar(value=0)

        fields = {
            "Name" : self.name_field,
            "Role" : self.role_field
        }

        for (k, v) in fields.items():
            f = Frame(form)
            f.pack(side="top", fill=X, expand=True)
            l = Label(f, padx=10, pady=10,  text=k, width=20, anchor=E)
            l.pack(side="left", fill=X, padx=0, pady=0)
            e = Entry(f, textvariable=v)
            e.pack(side="left", fill=X, expand=True, padx=4, pady=4, ipadx=6, ipady=6)

        f = Frame(form)
        f.pack(side="top", fill=X, expand=True)
        l = Label(f, padx=10, pady=10,  text="Admin", width=20, anchor=E)
        l.pack(side="left", fill=X, padx=0, pady=0)
        e = Checkbutton(f, variable=self.admin_field)
        e.pack(side="left", anchor=W)

        btns = Frame(form, pady=20, padx=10)
        save_btn = Button(btns, text="Save", command=self.on_save)
        cancel_btn = Button(btns, text="Cancel", command=self.on_cancel)
        save_btn.pack(side=RIGHT, padx=5)
        cancel_btn.pack(side=RIGHT, padx=5)
        btns.pack(side="top", fill=BOTH, expand=True)
        
    def present(self, profile=None):
        logger.info("Presenting profile edit form")
        self.current_profile = profile
        self.name_field.set(profile.name)
        self.role_field.set(profile.role)
        self.admin_field.set(profile.admin)
        super().present()

    def on_save(self):
        profile = self.current_profile
        profile.name = self.name_field.get()
        profile.role = self.role_field.get()
        profile.admin = self.admin_field.get()
        self.application.profiles.save(profile)
        self.application.edit_profiles()

    def on_cancel(self):
        self.application.edit_profiles()

    def on_back_button(self):
        self.application.edit_profiles()