from subprocess import check_call
from tkinter import Menu, StringVar


import logging
logger = logging.getLogger(__name__)

class LocalisedMenu(Menu):

    def __init__(self, parent, **kwargs):
        super().__init__(parent, tearoff=False, **kwargs)
        self.labels = []
        self.checks = []

    def insert_command(self, text_label, command=None, enable_check=None):
        self.add_command(command=command)
        self.labels.append(text_label)
        self.checks.append(enable_check)

    def insert_cascade(self, text_label, enable_check=None, **kwargs):
        self.add_cascade(**kwargs)
        self.labels.append(text_label)
        self.checks.append(enable_check)

    def render_labels(self):
        for i, label in enumerate(self.labels):
            enable_check = self.checks[i]
            state = "normal"
            if enable_check and not(enable_check()):
                state = "disabled"
            self.entryconfigure(i, label=_(label), state=state)



class Menubar(LocalisedMenu):
    
    def __init__(self, parent, application, **kwargs):
        super().__init__(parent, **kwargs)
        self.application = application
        self.submenus = []

        home_menu = self.newmenu()
        home_menu.insert_command("menu.home", command=application.show_home_screen, enable_check=self.has_active_profile)
        home_menu.insert_command("menu.switch_profile", command=application.show_profile_switcher, enable_check=self.has_active_profile)
        home_menu.insert_command("menu.exit", command=application.exit)
           
        self.insert_cascade("menu.scat", menu=home_menu)

        package_menu = self.newmenu()
        self.insert_cascade("menu.package", menu=package_menu, enable_check=self.has_active_profile)
        package_menu.insert_command("menu.package.new", command=application.new_pkg)
        package_menu.insert_command("menu.package.resume", command=application.resume_pkg)
        package_menu.insert_command("menu.package.open", command=application.open_pkg)
        package_menu.insert_command("menu.package.details", command=application.show_package_details)

        admin_menu = self.newmenu()
        self.insert_cascade("menu.admin", menu=admin_menu, enable_check=self.is_admin_profile)
        
        help_menu = self.newmenu()
        self.insert_cascade("menu.help", menu=help_menu)

        help_menu.insert_command("menu.help", command=self.application.open_help)
        help_menu.insert_cascade("menu.language", menu=self.build_langmenu())
        help_menu.insert_command("menu.about", command=self.application.show_about)
        self.help_menu = help_menu
        
        admin_menu.insert_command("menu.showconfig", command=self.application.show_config_screen)
        admin_menu.insert_command("menu.showencryption", command=self.application.show_encryption_config_screen)
        admin_menu.insert_command("menu.editprofiles", command=self.application.edit_profiles)
        admin_menu.insert_command("menu.addprofile", command=self.application.add_profile)
        admin_menu.insert_command("menu.showlogs", command=self.application.open_log_view)
        admin_menu.insert_command("menu.install_icon", command=self.application.install_desktop_icon)
        self.admin_menu = admin_menu

        self.on_lang_change()

    def build_langmenu(self):
        lang_var = StringVar()
        lang_var.set(self.application.current_language())
        language_menu = Menu(self, tearoff=False)
        for (langcode, name) in self.application.available_languages:
            language_menu.add_radiobutton(label=name, value=langcode, var=lang_var, command=self.switch_lang)
        self.lang_var = lang_var
        return language_menu

    def switch_lang(self):
        langcode = self.lang_var.get()
        logger.info("Switch language to %s", langcode)
        self.application.select_language(langcode)

    def on_lang_change(self):
        self.render_labels()
        for menu in self.submenus:
            menu.render_labels()
    
    def on_profile_change(self):
        self.on_lang_change()

    def newmenu(self):
        menu = LocalisedMenu(self)
        self.submenus.append(menu)
        return menu

    def has_active_profile(self):
        return self.application.current_profile != None

    def is_admin_profile(self):
        app = self.application
        return app.configonly or (app.current_profile and app.current_profile.admin)
