import json
import dataclasses

from uuid import uuid4
from dataclasses import dataclass, asdict
from pathlib import Path

import logging
logger = logging.getLogger(__name__)

@dataclass
class Profile:
    id: str
    name: str = ""
    role: str = ""
    lang: str = "en"
    admin: bool = False

    @property
    def org_label(self):
        return "(Administrator)" if self.admin else "(Normal user)"

    def __init__(self, **kwargs):
        names = set([f.name for f in dataclasses.fields(self)])
        for k, v in kwargs.items():
            if k in names:
                setattr(self, k, v)


class ProfileRepository():
    
    def __init__(self, json_path):
        self.json_path = json_path
        self.profiles = {}

    def list_all(self):
        return self.profiles.values()

    def list_users(self):
        return [profile for profile in self.list_all() if not(profile.admin)]

    def save(self, profile):
        self.profiles[profile.id] = profile
        if not(profile.admin) and self.count_admins() == 0:
            profile.admin = True
        self.persist()

    def count_admins(self):
        return [x.admin for x in self.profiles.values()].count(True)

    def new_profile(self):
        return Profile(id=uuid4().hex)

    def remove(self, profile):
        del self.profiles[profile.id]
        if self.count_admins() == 0:
            next_admin = next(iter(self.profiles.values()))
            next_admin.admin = True 
        self.persist()

    def is_singleusermode(self):
        users = self.list_users()
        return len(users) == 1
    
    def load(self):
        if self.json_path.exists():
            with open(self.json_path, 'r', encoding='utf-8') as f:
                c = f.read()
                state = json.loads(c)
                listing = state["profiles"]
                for item in listing:
                    profile = Profile(**item)
                    self.profiles[profile.id] = profile
        self.firstrun = len(self.profiles) == 0

    def persist(self):
        listing = [ asdict(profile) for (_, profile) in self.profiles.items() ]
        to_export = { "profiles": listing}
        with open(self.json_path, 'w', encoding='utf-8') as f:
            f.write(json.dumps(to_export))

@dataclass
class OrgConfig:
    name: str
    namespace: str
    archon_code: str


@dataclass
class EncryptionConfig:
    passphrase_file: str

class Configuration():
    
    def __init__(self, json_path):
        self.json_path = json_path
        self.data = {}
        self.orgconfig = OrgConfig("","","")
        self.encryption = EncryptionConfig("")

    def set_orgconfig(self, orgconfig):
        self.orgconfig = orgconfig
        self.persist()

    def set_encryption(self, encryption):
        self.encryption = encryption
        self.persist()

    def persist(self):
        logger.info("Persisting config")
        to_export = { 
            "institution": asdict(self.orgconfig),
            "encryption": asdict(self.encryption),
        }
        with open(self.json_path, 'w', encoding='utf-8') as f:
            f.write(json.dumps(to_export))

    def load(self):
        logger.info("Loading config from %s", self.json_path)
        if self.json_path.exists():
            with open(self.json_path, 'r', encoding='utf-8') as f:
                c = f.read()
                hydrated = json.loads(c)
                if "institution" in hydrated:
                    self.orgconfig = OrgConfig(**hydrated["institution"])
                if "encryption" in hydrated:
                    self.encryption = EncryptionConfig(**hydrated["encryption"])
            self.firstrun = False
        else:
            self.firstrun = True

    def is_encryption_configured(self):
        encryption = self.encryption
        logger.debug("Using passphrase file %s", encryption.passphrase_file)
        return encryption.passphrase_file and Path(encryption.passphrase_file).exists()
