import logging
import gettext
import glob

from pathlib import Path
from .workspace import LocalWorkspace
from .logs import LoggingConfiguration
from .config import ProfileRepository, Configuration
from .util import ChangeNotifier
from .packaging import PackageWorkspace
from scattool.shared import workspace
import datetime
import site

logger = logging.getLogger(__name__)

class ScatApp():

    def go(self):
        self.setup_root_dir()
        self.current_profile = None
        self.language_change_listeners = ChangeNotifier()
        self.profile_change_listeners = ChangeNotifier()
        self.languages = {}
        self.setup_logging()
        self.setup_i18n()
        self.setup_local_workspace()
        self.main()

    def setup_logging(self):
        self.logs = LoggingConfiguration(self.workspace_base / 'logs')
        self.logs.setup()
        

    def setup_i18n(self):
        self.available_languages = [ ]
        self.locales_folder = Path(site.getuserbase()) / 'scattool' / 'locales'
        logger.info("Using locale folder %s", self.locales_folder)
        for x in self.locales_folder.glob("*"):
            print("X")
            code = x.name
            try:
                t = gettext.translation('scattool', self.locales_folder, languages=[code], fallback=True)
                lang = t.gettext("_lang")
                self.available_languages.append( (code, lang) )
            except:
                logger.warn("Error loading %s language pack", code)
        logger.info("Discovered languages %s", self.available_languages)
        self.select_language('en')

    def get_resource_file(self, name):
        basepath = Path(site.getuserbase()) / 'scattool' / 'locales'
        path = basepath / self.langcode / name
        if path.exists():
            return path
        else:
            return basepath / "en" / name


    def select_language(self, code):
        logger.info("Configuring language to %s", code)
        self.langcode = code
        if not(code in self.languages):
            self.languages[code] = gettext.translation('scattool', self.locales_folder, languages=[code, 'en'], fallback=True)
        self.languages[code].install()
        self.language_change_listeners.notify()

    def add_languagelistener(self, listener):
        self.language_change_listeners.add_listener(listener.on_lang_change)

    def add_profilelistener(self, listener):
        self.profile_change_listeners.add_listener(listener.on_profile_change)

    def current_language(self):
        return self.langcode

    def setup_root_dir(self):
        folder_slug = ".scat"
        self.workspace_base = Path(Path.home() / folder_slug)

    def setup_local_workspace(self):
        logger.info("Using workspace %s", self.workspace_base)
        self.workspace = LocalWorkspace(self.workspace_base)
        self.workspace.initialise_folders()
        self.profiles = ProfileRepository(self.workspace.profilepath)
        self.profiles.load()
        self.configuration = Configuration(self.workspace.configfile)
        self.configuration.load()
        self.packaging_workspace = PackageWorkspace(self.workspace, self.configuration)

    def switch_profile(self, profile):
        logger.info("Switch profile to %s", profile)
        if (self.current_profile != profile):
            self.current_profile = profile
            self.profile_change_listeners.notify()
            self.on_profile_change()
        if self.packaging_workspace:
            logger.info("Setting current profile on workspace")
            self.packaging_workspace.switch_profile(self.current_profile)

    def on_profile_change(self):
        pass

    def encryption_enabled(self):
        return self.configuration.is_encryption_configured()

    def fresh_package_workspace(self):
        self.packaging_workspace = PackageWorkspace(self.workspace, self.configuration)
        self.packaging_workspace.switch_profile(self.current_profile)
        return self.packaging_workspace

    def open_local_package(self, type, package_identifier):
        logger.info("Looking up package location for %s %s", type, package_identifier)
        path = self.workspace.localstore.packagepath(type, package_identifier)
        self.open_package(path)

    def open_package(self, path):
        packaging_workspace = self.packaging_workspace
        package_identifier = path.name
        logger.info("Using %s", path)
        logger.info("Opening package %s", package_identifier)
        packaging_workspace.open_package(path)
        logger.info("Opened package %s", package_identifier)
        # underlying_package = packaging_workspace.underlying_package
        # if (underlying_package.doublebagged):
        #     underlying_package.passphrase_file = "/home/ccare/code/scat/scat-demo/ppf.txt"
        #     underlying_package.read_inner_bag()

        # print(underlying_package.report())

        # packaging_workspace.load_package_into_workspace()
        # # packaging_workspace.resume()
        # # packaging_workspace.clear()

        # packaging_workspace.save_as_aip(encryption=False, double_bag=False)
        # packaging_workspace.save_as_dip()
        # packaging_workspace.close_package()
