#! /usr/bin/python

## Incorporating code from these versions:
## 20210530
## version 0.1.0dev
## 20210613
## version 0.1.1dev

##  Sort out copyright
##  Sort out licence, Apache or Affero?


import csv
import json
from pathlib import Path
import re
import shutil
import subprocess

# import tarfile
# import tempfile
# import time
# import uuid

import scattool.sbagit as sbagit

import logging
logger = logging.getLogger(__name__)

##  logging, cli, how does this integrate with BagIt?

def _format_identification_for_digital_objects(path):
    logger.info("scatlib is doing format id for digital objects. Using path %s", path)
    if not path:
        print("fido error 1")
        return None
    if not Path(path).is_dir():
        print("fido error 2")
        return None

    fido = subprocess.run(
        [
            "python3",
            "-m",
            "fido.fido",
            "-q",
            "-recurse",
            "-zip",
            path
        ],
        capture_output = True,
        text=True
    )

    if fido.returncode:
        print("fido error", fido.returncode)
        return None
    
    pronom_formats = {}
    for line in fido.stdout.splitlines():
        reader = csv.reader([line])
        fields = list(reader)[0]
        if fields[0] == "OK":
            ##
            ##  check what fields[2] looks like
            ##  does it contain multiple entries
            ##  that require splitting?
            ##
            pronom_formats[fields[2]] = True

    if pronom_formats:
        formats = ",".join(sorted(pronom_formats.keys()))
    else:
        formats = ""
        
    return formats


class Scatlib:
    def __init__(self):
        super(Scatlib, self).__init__()

    def MakePackage(
        double_bag=None,
        encryption=None,
        encryption_algorithm=None,
        pass_phrase=None,
        pass_phrase_file_name=None,
        payload=None,
        adjunct_files=None,
        metadata=None,
        tar=None,
        zip=None,
        show_suffix=None,
    ):
        logger.info("Entering make package sequence")
        logger.info("Double bagging: %s", double_bag)
        logger.info("Encryption: %s", encryption)
        logger.info("Adjunct: %s", adjunct_files)
        logger.info("metadata: %s", metadata)
        logger.info("Container switches: tar=%s, zip=%s, show_suffix=%s", tar, zip, show_suffix)

        ## should do some validity tests


        ##  run fido and add format idntifiers to metadata
        logger.info("Starting format identification")
        formats = _format_identification_for_digital_objects(payload)
        logger.info("Completed format identification. Results were: %s", formats)

        tag = "Internal-Sender-Description"
        description = json.loads(metadata[tag])
        description["PRONOM formats"] = formats
        metadata[tag] = json.dumps(description)

        ## We used to flag when double bag, removed by Viv in 0.1.1
        ## if double_bag:
        ##    metadata["double_bag"] = double_bag
        
        package = sbagit.make_sbag(
            payload,
            bag_info=metadata,
            adjunct_files=adjunct_files,
            tar=tar,
            zip=zip,
            show_suffix=show_suffix,
        )

        ##  we should validate package ie serialized BagIt structure
        ##  validate
        sbag = sbagit.SBag(package.name)
        if sbag.is_valid:
            print("yay")
        else:
            print("boo")

        if encryption:
            ##  confirm pass phrase (incl via file) is present

            cipher = encryption_algorithm
            if not cipher:
                cipher = "AES256"

            ##  if both sources cited use file
            if Path(pass_phrase_file_name).is_file:
                ##  use pass_phrase
                # print("package.name", package.name)
                gpg = subprocess.run(
                    [
                        "gpg",
                        "--symmetric",
                        "--batch",
                        "--cipher-algo",
                        cipher,
                        "--passphrase-file",
                        pass_phrase_file_name,
                        package.name,
                    ]
                )
                if gpg.returncode:
                    ##  need to report error and fail
                    print(gpg)
                    print("###### fail #######")
            elif pass_phrase:
                pass

            ##  delete plain text original
            Path(package.name).unlink()

            ##  adjust file names
            package_file_name = package.name
            ##  attend to show suffix option
            if show_suffix:
                package_file_name += ".gpg"
            else:
                p = Path(package.name.with_suffix(".gpg"))
                shutil.move(p, p.with_suffix(""))

        if double_bag:
            package = sbagit.make_sbag(
                payload,
                bag_info=metadata,
                adjunct_files=adjunct_files,
                tar=tar,
                zip=zip,
                show_suffix=show_suffix,
            )
            ##  validate
            sbag = sbagit.SBag(package.name)
            if sbag.is_valid:
                print("yay_2")
            else:
                print("boo_2")

        return package

    def OpenPackage(file_name = None):
        sbag = sbagit.SBag(file_name)
        
        unpack = {}
        if sbag.is_valid():
            p = Path(sbag.path)
            unpack["name"] = p.name
            unpack["tags"] = sbag.tags
            unpack["algorithms"] = sbag.algorithms
            unpack["container"] = sbag.container
            unpack["metadata"] = sbag.info
            unpack["entries"] = sbag.entries
            unpack["adjunct"] = sbag.adjunct
            unpack["payload"] = sbag.payload

        fixity = {}
        for key in unpack["entries"].keys():
            if re.match("manifest-", key) or re.match("tagmanifest-", key):
                pass
            elif key == "bag-info.txt" or key == "bagit.txt":
                pass
            else:
                values = unpack["entries"][key]
                fixity[key.replace("data/", "", 1)] = values

        unpack["fixity"] = fixity
        
        return unpack
        

